;Using Microwire Peripherals with the Z8
;***************************************
;
;A common requirement of microntroller based applications is that the system
;have system have some form of analogue to digital conversion facility and/or a
;non-volatile storage element.  There are microcontrollers available which have
;these elements "on-chip".  The cost of these microcontrollers can often be
;prohibitive and a more cost effective solution is to use microwire peripherals
;to provide the functions externally to the microcontroller.  These peripherals
;are small in size and also low cost.
;
;This application note provides a listing of Z8 code which communicates with a
;microwire EEROM and an a/d converter.
;
;The source for the following routines is listed below, and the code is fully
;documented to describe the function being performed:
;
;       eerom_read - Read 16-bit data from the EEROM
;       eerom_write - Write 16-bit data to the EEROM
;       eerom_erase - Erase an EEROM address
;       eerom_lock - Lock out any EEROM writes
;       eerom_unlock - Unlock a locked EEROM
;       eerom_write_all - Write a 16-bit word to all EEROM addresses
;       eerom_erase_all - Erase all EEROM addresses
;       a2d_in - Read a value from an a/d channel
;
;The microwire devices used to develop the code were produced by National
;Semiconductor.
;
;This application used Z8 P0.6 as the serial clock output (SCLK), P0.0 as
;the data output from the Z8 (DIN on the microwire devices), and P0.7 as
;the data input to te Z8 (DOUT on the microwire devices).
;
;
;
;Register definitions
;
time_out:       .equ    r9
counter:        .equ    r10
rd_count:       .equ    r11
wr_count:       .equ    r12
output1:        .equ    r13
output2:        .equ    r14
input1:         .equ    r14
output3:        .equ    r15
input2:         .equ    r15



;***************************************************************
;
; This routine accesses the EEROM to perform a read.
; The address to be read is passed by the calling routine in the
; least significant four bits of r13 (output1).  The 16-bit data
; read from the EEROM is returned in r14 (input1, most sig byte)
; and r15 (input2, least sig byte).
;
;***************************************************************

eerom_read:     cp      time_out,#%00   ;is the time-out flag clear
                jr      nz,$-3          ;loop round if not
                and     output1,#%0f    ;clear the top four bits
                or      output1,#%80    ;set the ms bit for the EEROM read code
                ld      rd_count,#%10   ;want to read 16-bits
                ld      wr_count,#%08   ;want to write 8-bits
                jr      access_eerom


;***************************************************************
;
; This routine accesses the EEROM to perform a write.
; The address to be written is passed by the calling routine in
; the least significant four bits of r13 (output1).  The 16-bit
; data to be written to the EEROM is passed in r14 (output2, most
; sig byte) and r15 (output3, least sig byte).
;
;***************************************************************

eerom_write:    cp      time_out,#%00   ;is the time-out flag clear
                jr      nz,$-3          ;loop round if not
                and     output1,#%0f    ;clear the top four bits
                or      output1,#%40    ;set bit d6 for the EEROM write code
                jr      write_24


;***************************************************************
;
; This routine erases the data at an EEROM address.
; The address is passed to this routine in the bottom four bits
; of r13 (output1).
;
;***************************************************************

eerom_erase:    cp      time_out,#%00   ;is the time-out flag clear
                jr      nz,$-3          ;loop round if not
                and     output1,#%0f    ;clear the top four bits
                or      output1,#%c0    ;set bits d7 & d6 for the EEROM erase code
                jr      write_8


;***************************************************************
;
; This routine locks the EEROM from being able to be written or
; erased.
;
;***************************************************************

eerom_lock:     cp      time_out,#%00   ;is the time-out flag clear
                jr      nz,$-3          ;loop round if not
                ld      output1,#%00    ;EEROM lock code is four zeros
                jr      write_8


;***************************************************************
;
; This routine unlocks the EEROM, and allows it to be written or
; erased.
;
;***************************************************************

eerom_unlock:   cp      time_out,#%00   ;is the time-out flag clear
                jr      nz,$-3          ;loop round if not
                ld      output1,#%30    ;EEROM unlock code is 0011XXXX
                jr      write_8



;***************************************************************
;
; This routine writes the data held in r14 (output2 - ms byte)
; and r15 (output3 - ls byte) to all EEROM address locations
;
;***************************************************************

eerom_write_all:        cp      time_out,#%00   ;is the time-out flag clear
                jr      nz,$-3          ;loop round if not
                ld      output1,#%10    ;set up the EEROM write all code
write_24:       ld      wr_count,#%18   ;want to write 24-bits
                ld      rd_count,#%00   ;don't want to read any bits
                jr      access_eerom


;***************************************************************
;
; This routine erases all EEROM address locations
;
;***************************************************************

eerom_erase_all:        cp      time_out,#%00   ;is the time-out flag clear
                jr      nz,$-3          ;loop round if not
                ld      output1,#%20    ;EEROM erase all code is 0010XXXX
write_8:        ld      wr_count,#%08   ;want to write 8-bits
                ld      rd_count,#%00   ;don't want to read any bits
access_eerom:   and     P0,#%bf         ;set the SCLK line low
                or      P0,#%20         ;set the eerom chip select high
                nop
                or      P0,#%40         ;pulse SCLK high and low coz the
                                        ;eerom needs this
                nop
                and     P0,#%bf
                call    access_uwire
                and     P0,#%df         ;set the eerom chip select low
                ld      time_out,#%20   ;set up the delay needed for after
                                        ;the eerom access
                ret


;***************************************************************
;
; This routine accesses the a/d channel held in the bottom
; three bits of r13 (output1).  The 8-bit digital value is
; returned in r14 (input1).
;
;***************************************************************

a2d_in:         and     output1,#%07    ;clear the top five bits
                or      output1,#%08    ;set sgl/dif bit
                swap    output1         ;get address in top nibble
                ld      rd_count,#%08   ;want to read 8-bits
                ld      wr_count,#%04   ;want to write 4-bits
                and     P0,#%af         ;set the a/d chip select & SCLK line low
                call    access_uwire
                or      P0,#%10         ;set the a/d chip select high
                ret


;***************************************************************
;
; This routine exercises the microwire interface according
; to the values found in working registers r11 .. r15.
;
;       r11 = No. of bits to read               (rd_count)
;       r12 = No. of bits to write              (wr_count)
;       r13 = C0,C1,C2,C3,D0,D1,D2,D3           (output1)
;       r14 = D4,D5,D6,D7,D8,D9,D10,D11         (output2)
;       r15 = D12,D13,D14,D15,D16,D17,D18,D19   (output3)
;
; where C0 to C4 are the the four control bits clocked out,
; and D0 to D19 are the address/data bits clocked out. The
; actual number of address/data bits used is defined by the
; value in r12.  If the value in r11 is zero, no data is
; clocked in.  For a non-zero value, the appropriate number
; of bits are returned in registers r14 and r15 (e.g. msb in
; bit D7 of r14, lsb in bit D0 of r15 for a 16-bit word).
;
; The code requires that the appropriate chip select line be
; activated by the calling routine, and that the microwire
; peripheral is waiting for the first data bit.  Any differences
; in the start of the cycle timing for the different peripherals
; has to be catered for by the calling routine.  The chip select
; also has to be de-activated after return.
;
; So, SCLK is low when we enter this routine and we can write
; the first data bit to the peripheral.
;
;***************************************************************

access_uwire:   di                      ;don't want interrupts during this
                or      P0,#%80         ;set peripherals DIN line for the start bit
                or      P0,#%40         ;send the SCLK line high
                nop
                and     P0,#%bf         ;send the SCLK line low
                ld      counter,#%08    ;initialise the counter

; register "output1" contans the first 8 bits to be clocked out.

first_8:        rl      output1         ;rotate msb into carry
                jr      c,set_data1     ;set the peripherals DIN line if bit was a 1
                and     P0,#%7f         ;if 0, set the DIN line to a 0
                jr      clock1          ;now jump to clock the bit in
set_data1:      or      P0,#%80         ;set the DIN line to a 1
clock1:         or      P0,#%40         ;send the SCLK line high
                nop
                and     P0,#%bf         ;send the SCLK line low
                djnz    counter,chk4mor1 ;check write count if not yet sent
                                            ;8 bits from "output1"
                jr      second_8         ;if have done 8 bits move on to
                                         ;next register
chk4mor1:       djnz    wr_count,first_8 ;loop round if we haven't sent enough
                jr      check4rd         ;if we have sent enough bits then see
                                        ;if we need to read anything.

; If we get here, we must have had to send more than 8 bits.
; Bits 9 to 16 are in register "output2".

second_8:       ld      counter,#%08    ;initialise the counter for the next 8 bits
                djnz    wr_count,second_loop ;loop round if we haven't sent enough
                jr      check4rd        ;if we have sent enough bits then see
                                ;if we need to read anything.

second_loop:    rl      output2         ;rotate msb into carry
                jr      c,set_data2     ;set the DIN line if bit was a 1
                and     P0,#%7f         ;if 0, set the DIN line to a 0
                jr      clock2          ;now jump to clock the bit in
set_data2:      or      P0,#%80         ;set the DIN line to a 1
clock2:         or      P0,#%40         ;send the SCLK line high
                nop
                and     P0,#%bf         ;send the SCLK line low
                djnz    counter,chk4mor2 ;check write count if not yet sent
                                            ;8 bits from "output2"
                jr      third_8          ;if have done 8 bits move on to
                                         ;next register
chk4mor2:       djnz    wr_count,second_loop ;loop round if we haven't sent enough
                jr      check4rd         ;if we have sent enough bits then see
                                ;if we need to read anything.

; If we get here, we must have had to send more than 16 bits.
; Bits 17 to 24 are in register "output3".

third_lp:       rl      output3         ;rotate msb into carry
                jr      c,set_data3     ;set the DIN line if bit was a 1
                and     P0,#%7f         ;if 0, set the DIN line to a 0
                jr      clock3          ;now jump to clock the bit in
set_data3:      or      P0,#%80         ;set the DIN line to a 1
clock3:         or      P0,#%40         ;send the SCLK line high
                nop
                and     P0,#%bf         ;send the SCLK line low
third_8:        djnz    wr_count,third_lp ;loop round if we haven't sent enough

; Now see if we are required to read any data from the microwire
; peripheral.

check4rd:       inc     rd_count        ;fudge the read count for the next bit
                djnz    rd_count,do_read ;read data if rd_count is not zero
                jr      tidy_up         ;get ready to go back to calling routine

do_read:        clr     input1          ;initialise the data registers
                clr     input2
                ld      counter,#%08    ;initialise the counter
read_it1:       or      P0,#%40         ;send the SCLK line high
                nop
                and     P0,#%bf         ;send the SCLK line low
                nop
                ld      wr_count,P0     ;use wr_count as temp reg
                rr      wr_count        ;shift DOUT into carry
                rlc     input1          ;shift the bit into lsb of 1st data reg
                djnz    counter,more_rd1 ;more bits need to be shifted into
                                           ;input1 if not zero
                jr      do_read2        ;input1 full, start on input2
more_rd1:       djnz    rd_count,read_it1 ;round again if more to do
                jr      tidy_up

more2rd:        or      P0,#%40         ;send the SCLK line high
                nop
                and     P0,#%bf         ;send the SCLK line low
                nop
                ld      wr_count,P0     ;use wr_count as temp reg
                rr      wr_count        ;shift DOUT into carry
                rlc     input2          ;shift the bit into lsb of 1st data reg
                djnz    counter,read_it2 ;more bits need to be shifted into
                                           ;input2 if not zero
                jr      tidy_up         ;input2 full, go home
do_read2:       ld      counter,#%08    ;initialise the counter
read_it2:       djnz    rd_count,more2rd ;round again if more to do
tidy_up:        ei
                ret


