;========================================================================
;
;       ZILOG SUPER8 "MICRO-TALKER" MONITOR PROGRAM
;       (c) 1989,1990,1991 T-Recursive Technology
;       placed into the public domain for free and unrestricted use
;
;       A minimal monitor program for the Zilog Super8.
;       Used in conjunction with a "smart" host program to examine &
;       modify code memory, external memory, and registers, and
;       to set and execute breakpoints in machine language and Forth.
;
;       The Talker program uses only 6 registers (BAh to BFh here),
;       no RAM (although some versions hold stack in RAM), and about
;       512 bytes of PROM.  No interrupts are used.  The Talker may
;       be operated half-duplex over a bidirectional serial line.
;
;       The program accepts characters from the Super8 UART.
;       Characters 30h to 3Fh are treated as hex digits and are
;       shifted into a one-byte data register.
;       Characters 20h to 2Fh are command codes:
;           20-23 reserved for future use (ignored)
;         group 2: breakpoint / debug support
;           24 = set a Forth "thread" breakpoint
;           25 = set a Forth "code field" breakpoint
;           26 = set a machine language breakpoint
;           27 = fetch lo byte of address (adrs lo -> data & output)
;           28 = fetch hi byte of address (adrs hi -> data & output)
;           29 = read back data register  (data -> output)
;         group 1: minimal talker
;           2A = fetch byte & incr addr   (mem or reg -> data & output)
;           2B = store byte & incr addr   (data -> mem or reg)
;           2C = set lo byte of address   (data -> adrs lo)
;           2D = set hi byte of address   (data -> adrs hi)
;           2E = set memory "page"        (data -> page)
;           2F = "go"                     (jump to current adrs)
;
;       The memory "page" is interpreted as follows for store and fetch:
;           0 = C (code) memory
;           1 = E (external) memory
;           2 = indirect registers 00-FF
;           3 = direct registers 00-FF, bank 0
;           4 =   "        "       "  , bank 1
;       The difference between "indirect" and "direct" register access
;       is only apparent for those registers between C0 and FF.
;
;       Internal register usage:
;           rr14 (BE,BF) = address
;           r13  (BD)    = memory "page"
;           r12  (BC)    = data byte
;           r11  (BB)    = working
;           r10  (BA)    = saved rp1
;
;       Revision history
;       v 1.0  23 Aug 89    original program; functions 29h - 2Fh;
;               access to memory, registers, indirect registers;
;               'talker' and 'talk' entry points.
;
;       v 1.1  25 Feb 90    support for breakpoints and Forth words;
;               standalone initialization.
;
;       v 1.2  7 May 90     function codes reassigned; added address
;               readback; improved breakpoint support; fixed problem
;               with direct register access to C8-CF and to RP1...now
;               correctly uses application program's registers.
;
;       v 1.2CP  7 Jun 90   modified for Teatronics Echelon Channel Proc,
;               as an include file.  Uses half-duplex RS-485.  Sets P37
;               low to turn LED 'on'.  Pulses watchdog on TXD.
;               Fixed bug where tx turned off during last tx character.
;               Changed to disable irpts on 'mbreak' entry, enable on 'go'.
;
;               26 Jun 90   altered to assemble standalone; set IMR to 00
;               so that irpt enable on 'go' doesn't lock up system.
;
;               4 Aug 90  fixed p0 init value; use int'l regs for stack
;
;               17 Oct 90  moved stack to FFB0h so program would work
;               in Exec Proc (where FFF8-FFFF is I/O)
;
;       v 1.3  22 Aug 90   changed direct register access so as not to need
;               writeable "Code" memory - now does direct register access
;               by changing RP1.
;
;       v 1.3vp  18 Aug 90  modified for Echelon Video Processor
;               using full duplex rs-485, no watchdog 'kick', stack
;               in RAM at 7f00, irpt vecs at 300h for prom emulator debug.
;
;       v 1.3xp 10 Oct 90  modified for SCC port A on Echelon ExecProc.
;
;       v 1.4   10 Feb 91  combined various talker versions into single
;               conditionally-assembled file.  Added fixed entry points
;               into 'talk' and 'mbreak'.  Changed 'talk' to return
;               application's rp1 value.
;
;       Known bugs & peculiarities
;       1.  registers C0-CF:  *** changed from version 1.2 ***
;           Doing a "direct" (bank 0 or 1) access to registers C0-CF
;           will reach the Super8's PHYSICAL registers C0-CF.  The
;           application's mapping through RP0 and RP1 is IGNORED.
;           If you wish to examine the application's "logical" C0-CF,
;           you must manually compute the physical address from the
;           contents of RP0 or RP1.
;
;       2.  access to RP1:  RP1 is used for direct register access, and so
;           CANNOT be directly altered.  To fetch or store the application's
;           RP1 value, you must use the 'rp1sv' (rp1-save) register.
;           (Physical register 0BAh in this version of the talker.)
;
;========================================================================
;
;       Monitor Configuration and Assembly-Time Options
;
;       STANDALONE - set to indicate the level of initialization.
;           0: no initialization (use when INCLUDEing in another program)
;           1: standalone operation in PROM
;           2: standalone initialization, but transfer to an application
;
standalone .equ 0

;
;       SYSTEM - set to indicate the hardware configuration this
;           talker is to be assembled for:
;           0: "generic" Super8 talker, stack in registers
;           1: CP talker, stack in RAM, half-duplex I/O
;           2: VP talker, stack in RAM, full-duplex I/O
;           3: XP talker, stack in RAM, I/O thru SCC
;           4: PP talker, stack in registers, 13 bit addressing
;

; system  .equ    2   ; Echelon VP

;========================================================================
;
;       MACROS
;
;       Macros for half-duplex communication on a bidirectional link,
;       e.g., a bidirectional RS-485 serial line.  Define these macros
;       to control the transceiver connected to the Super8's serial port.
;       If full-duplex is to be used (separate transmit and receive data
;       lines), define these as "null" macros.
;
;       TXON - turns serial port transmitter on, and receiver off.
;       TXOFF - turns serial port transmitter off, and receiver on.
;       KICK - kick the watchdog timer (v. 1.2CP)
;
;========================================================================

	.if system=1 ; CP ---------------------------------------------
txon    .macro  ; v. 1.2CP
	clr     wkg             ; wait a bit for host to turn its tx off
	dec     wkg
	jr      nz,$-2
	or      p2,#40h         ; p26 hi = tx
	.endm

txoff   .macro  ; v. 1.2CP
	clr     wkg             ; wait ~1 msec for last char to clear tx
	dec     wkg
	nop
	nop
	nop
	nop
	jr      nz,$-6
	and     p2,#0bfh        ; p26 lo = rx
	.endm

kick    .macro  ; v. 1.2CP, kick the watchdog timer
	sb0     ; use only when tx empty!
	and     utc,#7fh        ; p31 is pgm'd output
	xor     p3,#2           ; toggle p31
	xor     p3,#2           ; toggle p31
	or      utc,#80h        ; p31 is tx data
	.endm
	.endif ; CP ---------------------------------------------------

	.if system=2 || system=0 ; VP or generic ----------------------
txon    .macro  ; v. 1.3VP
	.endm

txoff   .macro  ; v. 1.3VP
	.endm

kick    .macro  ; v. 1.3VP   watchdog pin 6 lifted on VP
	.endm
	.endif ; VP ---------------------------------------------------


	.if system=3 ; XP ---------------------------------------------
ds1215  .equ    0fff9h  ; DS1215 timechip / SCC data ports A and B READ
	  ; and 0fffbh  ; disable SCC receivers before using!
scc     .equ    0fffch  ; Z8530 Serial Communications Ctrlr     "LDC"
sccacmd .equ    scc+2           ; command register, port A
sccadata .equ	scc+3		; data register, port A

fast    .macro
	and     emt,#10000011b  ; fast, no waits; wait\,dma,stack N/C
	.endm

sccslow .macro                  ; Z8530 SCC -
	or      emt,#00100000b  ; fast, 2 "C" waits
	.endm

txon    .macro  ; v. 1.3XP
	.endm

txoff   .macro  ; v. 1.3XP
	.endm

kick    .macro  ; v. 1.3XP, kick the watchdog timer
	sb0     ; use only when tx empty!              CP and XP
	and     utc,#7fh        ; p31 is pgm'd output      |
	xor     p3,#2           ; toggle p31               |
	xor     p3,#2           ; toggle p31               |
	or      utc,#80h        ; p31 is tx data           v
	.endm
	.endif ; XP ---------------------------------------------------

	.if system=4 ; PP ---------------------------------------------
txon    .macro  ; v. 1.3PP
	clr     wkg             ; wait a bit for host to turn its tx off
	dec     wkg
	jr      nz,$-2
	or      p0,#40h         ; p06 hi = tx   PP
	.endm

txoff   .macro  ; v. 1.3PP
	clr     wkg             ; wait ~1 msec for last char to clear tx
	dec     wkg
	nop
	nop
	nop
	nop
	jr      nz,$-6
	and     p0,#0bfh        ; p06 lo = rx   PP
	.endm

kick    .macro  ; v. 1.3PP, kick the watchdog timer
	xor     p2,#80h         ; toggle p27            PP
	xor     p2,#80h         ; toggle p27            "
	.endm
	.endif ; PP ---------------------------------------------------

; ========================================================================
;
;       MEMORY EQUATES
;
; ========================================================================
;
;       Location of RAM used by monitor: the monitor requires up to
;       8 bytes of stack space.
;
	.if system=0 ; generic
stack:  .equ    0       ; high registers
initemt .equ    7ch     ; no WAIT\, slow mem, 3 waits, stack & dma in regs
initp0m .equ    0ffh    ; enable 16 bit address output
	.endif

	.if system=1 ; CP
stack:  .equ    0ffb0h  ; 'E' mem
initemt .equ    03h     ; no WAIT\, fast mem, 0 waits, stack & dma in mem
initp0m .equ    0ffh    ; enable 16 bit address output
	.endif

	.if system=2 ; VP
stack:  .equ    7f00h   ; 'E' mem in lower 32K
initemt .equ    17h     ; no WAIT\, fast mem, 1 wait, stack & dma in mem
initp0m .equ    0ffh    ; enable 16 bit address output
	.endif

	.if system=3 ; XP
stack   .equ    0ff00h  ; 'E' mem, avoiding I/O devices
initemt .equ    17h     ; no WAIT\, fast mem, 1 wait, stack & dma in mem
initp0m .equ    0ffh    ; enable 16 bit address output
	.endif

	.if system=4 ; PP
stack:  .equ    0       ; high registers
initemt .equ    14h     ; no WAIT\, fast mem, 1 wait, stack & dma in regs
initp0m .equ    3fh     ; enable 14 bit address output (PP)
	.endif

;
;       Register bank used by monitor: the monitor requires 6 bytes
;       of Super8 registers.  These must be the last 6 bytes of an
;       8-byte register bank, since they will be mapped (via RP1) onto
;       working registers R11-R15.  These will be accessed directly
;       as working registers AND as general purpose registers, so
;       they MUST be located in the register range 00-BF.
;       Set 'regs' to an 8-byte boundary between 00 and B8, inclusive.
;
regs:   .equ    0b8h    ; monitor will use regs+3 to regs+7 (BB to BF)

rp1sv:  .equ    regs+2  ; save area for applic's rp1
wkg:    .equ    regs+3  ; working "scratch" register    aka r11
mdr:    .equ    regs+4  ; memory data register          aka r12
mbr:    .equ    regs+5  ; memory bank register          aka r13
mar:    .equ    regs+6  ; memory address register pair  aka rr14

;========================================================================
;
;       STANDALONE INITIALIZATION
;
;       For use when the monitor is used as a standalone program
;       in a Super8 development board.  In this case, the monitor
;       is located in low PROM, to be started on reset.  The Super8
;       registers are "minimally" initialized, to allow full-duplex
;       serial communication at 9600 baud, 8 bits, no parity.
;       The interrupts are vectored to a supplementary jump table
;       in code RAM.
;
;       Expects:  reset state for all Super8 mode & control registers
;       Returns:
;       Uses:
;
;========================================================================
	.if standalone

	.if standalone=2
vecs    .equ    0300h   ; jump table for an application at 330h
	.else
vecs    .equ    08000h  ; base address of interrupt jump table in code RAM
	.endif

	.org    0       ; the Super8 interrupt vector table
	.word   vecs,   vecs+3, vecs+6, vecs+9,
	.word   vecs+12,vecs+15,vecs+18,vecs+21
	.word   vecs+24,vecs+27,vecs+30,vecs+33
	.word   vecs+36,vecs+39,vecs+42,vecs+45

	jr      reset   ; fixed entry point 20h
	jp      talk    ; fixed entry point 22h
	jp      mbreak  ; fixed entry point 25h

;
;       Reset entry point for Super8 (address 20h)  Echelon Panel Processor
;
reset:  di
	sb0
	; the following instructions duplicate the Super8 reset conditions
	srp     #0c0h           ; default rp0 = 0c0h, rp1 = 0c8h
	ld      sym,#0          ; no 3-state, no irpts enabled

	; the following instructions are required
	ld      imr,#0          ; disable all interrupt levels
	ld      ipr,#0eeh       ; priority 7>6>5>4>3>2>0>1
	clr     p0              ; hi address is zero!             3 Aug 90
	nop
	nop
	nop
	ld      pm,#28h         ; P1 adr/dta, P0 out, push-pull, DM\ enbl
	nop
	nop
	nop
	ld      p0m,#initp0m    ; enable 14/16 bit address output

	ldw     sph,#stack      ; first push will use reg ?? / mem ????
	ld      emt,#initemt    ; wait, mem speed, stack & dma mem/reg

; this initializes the UART and its I/O bits

	.if system=0 ; generic
	ld      p2am,#80h       ; p31(txd) out, p30(rxd) in, p21 p20 in
	ld      p2cm,#80h       ; p35(dm\) out, p34 p25 p24 in
	ld      p2dm,#00h       ; p37 p36 p27 p26 in
	sb1
	ldw     ubgh,#0fh       ; 9600 baud w/ 20 MHz (19.6608 MHz) xtal
	ld      uma,#70h        ; x16, 8 bits, no parity
	ld      umb,#1eh        ; BRG rx/tx clk, BRG internal, BRG enable
	sb0
	ld      utc,#88h        ; P31 txd, 1 stop bit, enable tx
	ld      urc,#02h        ; enable rx
	ld      uie,#0          ; no uart interrupts enabled
	.endif

	.if system=1 ; CP
	ld      p2am,#80h       ; p31(txd) out, p30(rxd) in, p21 p20 in
	ld      p2cm,#80h       ; p35(dm\) out, p34 p25 p24 in     (CP)
	ld      p2dm,#83h       ; p37 out, p36 p27 in, p26 o.d.    (CP)
	sb1
	ldw     ubgh,#0fh       ; 9600 baud w/ 20 MHz (19.6608 MHz) xtal
	ld      uma,#70h        ; x16, 8 bits, no parity
	ld      umb,#1eh        ; BRG rx/tx clk, BRG internal, BRG enable
	sb0
	ld      utc,#88h        ; P31 txd, 1 stop bit, enable tx
	ld      urc,#02h        ; enable rx
	ld      uie,#0          ; no uart interrupts enabled
	.endif

	.if system=2 ; VP
	ld      p2am,#10000000B ; p31 out, p30 p21 p20 in               (VP)
	ld      p2bm,#00010101B ; p33 in, p32 irpt, p23 p22 initially irpt|
	ld      p2cm,#10000000B ; p35 out, p34 p25 p24 in                 |
	ld      p2dm,#10100000B ; p37 p36 out, p27 p26 in                 |
	ld      p2,#0ffh        ; bus ctl hi when enabled                 |
	ld      p3,#0ffh        ; led off for now, pr STRB hi             v
	sb1
	ldw     ubgh,#0fh       ; 9600 baud w/ 20 MHz (19.6608 MHz) xtal
	ld      uma,#70h        ; x16, 8 bits, no parity
	ld      umb,#1eh        ; BRG rx/tx clk, BRG internal, BRG enable
	sb0
	ld      utc,#88h        ; P31 txd, 1 stop bit, enable tx
	ld      urc,#02h        ; enable rx
	ld      uie,#0          ; no uart interrupts enabled
	.endif

	.if system=3 ; XP
	ld      p2am,#10001001B ; p31 out, p30 in, p21 out, p20 irpt    (XP)
	ld      p2bm,#01010101B ; p33 p32 irpt, p23 p22 initially irpt    |
	ld      p2cm,#10010000B ; p35 out, p34 irpt, p25 p24 in           |
	ld      p2dm,#10100011B ; p37 p36 out, p27 in, p26 o.d.           |
	ld      p2,#0bfh        ; panel tx disbl, bus ctl hi when enabled |
	ld      p3,#0bfh        ; led off for now, fdc side 0             v
	sb1
	ldw     ubgh,#0fh       ; 9600 baud w/ 20 MHz (19.6608 MHz) xtal
	ld      uma,#70h        ; x16, 8 bits, no parity
	ld      umb,#1eh        ; BRG rx/tx clk, BRG internal, BRG enable
	sb0
	ld      utc,#88h        ; P31 txd, 1 stop bit, enable tx
	ld      urc,#02h        ; enable rx
	ld      uie,#0          ; no uart interrupts enabled
	.endif

	.if system=4 ; PP
	ld      p2,#0ffh        ; all select outputs high
	ld      p3,#0ffh        ; all panel adr outputs high
	ld      p2am,#80h       ; p31(txd) out, p30(rxd) in, p21 p20 in
	ld      p2bm,#10100000b ; p33 p32 out, p23 p22 (whl) in    (PP)
	ld      p2cm,#10101010b ; p35(dm\) out, p34 p25 p24 out    (PP)
	ld      p2dm,#10001010b ; p37 out, p36 in, p27 p26 out     (PP)
	sb1
	ldw     ubgh,#0fh       ; 9600 baud w/ 20 MHz (19.6608 MHz) xtal
	ld      uma,#70h        ; x16, 8 bits, no parity
	ld      umb,#1eh        ; BRG rx/tx clk, BRG internal, BRG enable
	sb0
	ld      utc,#88h        ; P31 txd, 1 stop bit, enable tx
	ld      urc,#02h        ; enable rx
	ld      uie,#0          ; no uart interrupts enabled
	.endif

	.if system=3 ; XP
;========================================================================
;
;	Z8530 SCC INITIALIZATION
;
;	This routine is the "standard" initialization for the
;	Zilog Serial Communications Controller.  It follows
;	the recommended initialization sequence set forth in
;	the Zilog Z8530 SCC Technical Manual, Chapter 8.
;
;========================================================================
;
;	SCCINIT - initialize Z8530 SCC
;
;	Expects: none
;	Returns: none
;	Uses:	 r8-r13
;
sccinit: sccslow
	ldw     rr8,#sccacmd    ; initialize port A  (rs232)
	ldw	rr10,#sccaparms
	ld	r13,#numaparms
	call	sccfill
	fast

	.endif ; XP

	.if standalone=2
	jp      330h            ; jump to an application at 330h
	.else
	jr      talker          ; enter the talker program
	.endif

	.endif ; standalone

;========================================================================
;
;       TALKER MAIN ENTRY POINT
;       MACHINE LANGUAGE BREAKPOINT ENTRY
;
;       This is the main "talker" program.  It calls the basic
;       character processing routine "talk" repeatedly, until a monitor
;       command transfers control to an application program.
;
;       This is also the entry point for machine language breakpoints.
;       A breakpoint consists of a "CALL" to this address.  The call
;       causes the calling address to be pushed on the stack; this
;       routine pushes the flags as well, to allow a common routine to
;       service both CALLs and interrupts.
;
;       Should it be desirable to have an interrupt cause a breakpoint
;       -- e.g., a "break" pushbutton wired to an interrupt input --
;       the alternate "ibreak" entry point can be used.
;
;       The breakpoint routine copies the saved flag values into the
;       talker's data register, and the saved return address into the
;       talker's address register.  An immediate "go" function will
;       resume with these saved values (as well as the saved rp1).
;
;       Entering the breakpoint routine causes the '*' character to
;       be sent to the host.  Entering the monitor causes 'M' to be sent.
;       Note also that the main entry point its own address onto
;       the stack; this is so that we can "go" to a routine which ends
;       in a RET.  This "normal termination" can be distinguished from
;       a breakpoint by the 'M' character.
;
;       Expects:
;       Returns:
;       Uses:   4 bytes of stack; + 2 bytes if direct register access used
;
;========================================================================
	.begin
~pushme: call   ~t0
talker: ld      wkg,#'M' ; Talker program entry point
	jr      ~pushme     ; some cleverness to push the address 'talker'

mbreak: push    flags   ; Machine language breakpoint entry
	di
ibreak:                 ; Interrupt breakpoint entry
	pop     mdr         ; get saved flags in mdr
	pop     mar         ; get saved adrs in mar (2 bytes)
	pop     mar+1
	ld      wkg,#'*'

	; All entry points eventually land here
~t0:
	sb0
	and     p3,#07fh ; p37 lo = led on  (v. 1.2CP)

	.if system<>3 ; not XP, use s8 uart ---------------------------
~t1:    tm      utc,#2   ; previous char (if any) done?
	jr      z,~t1
	push    wkg      ; save the 'M' or '*' character during 'txon'
	txon
	pop     uio      ; transmit 'M' or '*' to signal monitor/brkpt
~t2:    tm      utc,#2   ; wait 'til character finished
	jr      z,~t2
	txoff

	.else ; XP ----------------------------------------------------
	ld      rp1sv,rp1   ; save application's rp1 value
	srp1    #regs    ; point to talker wkg regs
	sccslow
	push    wkg      ; save the 'M' or '*' character
~t1:    ldc     r11,sccacmd
	tm      r11,#4   ; previous char (if any) done?
	jr      z,~t1
	txon
	pop     wkg      ; transmit 'M' or '*' to signal monitor/brkpt
	ldc     sccadata,r11

~t2:    ld      r11,#1   ; wait until tx empty
	ldc     sccacmd,r11
	nop
	nop
	nop
	ldc     r11,sccacmd
	tm      r11,#1
	jr      z,~t2
	txoff
	fast
	ld      rp1,rp1sv   ; restore application's rp1 before 'talk'
	.endif ; XP ---------------------------------------------------

~t3:    kick            ; at this point tx is off, so kick TXD  v. 1.2CP
	call    talk    ; the talker loop
	jr      ~t3
	.end

;========================================================================
;
;       FORTH LANGUAGE BREAKPOINT ENTRIES
;
;       These are the entry points for Forth language breakpoints.
;
;       The first is the "code field" breakpoint.  This is an address
;       which can be stored in a Forth word's code field, to cause a
;       break whenever that word is executed.  This kind of breakpoint
;       can be set in any Forth word.
;       >>> In the Super8 Direct-Threaded-Code implementation, the
;         parameter field of every Forth word begins with machine code.
;         So, an ordinary machine-code breakpoint can be set in the
;         first 3 bytes of a word.  (All words have at least 3 bytes.) <<<
;
;       The second kind is the "thread" breakpoint.  This is an address
;       which can be patched into a high-level "thread", to cause a
;       break when a certain point is reached in high-level code.  The
;       thread is a series of addresses of Forth words, executed by the
;       inner or "NEXT" interpreter.  So, we provide the address of a
;       dummy Forth word whose execution action is to invoke a machine
;       language breakpoint.  The Forth execution state can be deduced
;       from the registers.
;
;========================================================================
cbreak: .equ    mbreak  ; the code field breakpoint is simply
			; a machine language breakpoint set
			; in a Super8 DTC "code field"

; what follows is the parameter and code field of a "headerless"
; Forth word, to invoke the breakpoint routine.  (In DTC, this is
; simply a machine code fragment which does a breakpoint.)

tbreak: call    mbreak  ; the thread breakpoint is simply
			; a pointer to this code fragment

;========================================================================
;
;       TALK - SINGLE CHARACTER PROCESSING ROUTINE
;
;       This routine processes one character received from the host.
;       If no character is received, it returns immediately.
;       It may cause two characters to be transmitted to the host.
;
;       This routine is called repeatedly in a tight loop from the
;       'talker' program, if invoked standalone or by a breakpoint.
;       It may also be called repeatedly from within an application
;       program, as a polled "background" task, to perform monitor
;       functions simultaneously with the application.
;
;       Note - on entry, the stack contains the following:
;                       return adrs in 'talker', lo
;                SP-->  return adrs in 'talker', hi
;
;       Expects:
;       Returns: rp1 preserved, bank 0 selected
;       Uses:    rp1, flags, regs BBh-BFh.
;
;========================================================================
	.begin
talk:   sb0                     ; check for received character
	ld      rp1sv,rp1       ; save application's rp1 value
	srp1    #regs           ; point r8-r15 to talker's regs

	.if system<>3 ; not XP ----------------------------------------
	tm      urc,#1
	jr      z,~exit
	ld      r11,uio         ; get character

	.else ; XP ----------------------------------------------------
	sccslow
	ldc     r11,sccacmd
	tm      wkg,#1
	jr      z,~exit
	ldc     r11,sccadata    ; get character

	.endif ; XP ---------------------------------------------------

	and     r11,#3fh        ; mask off all but low 6 bits
	sub     r11,#30h        ; if less than 30 -
	jr      ult,~cmd        ; - it's a command

;
; 30-3Fh: digit entry
;
~digit: swap    r12             ; 30-3f: hex digit, shift into lo nybble
	and     r12,#0f0h
	add     r12,r11         ;   note that r11 has been converted to 00-0f

~done:
~exit:
	.if system=3 ; XP
	fast
	.endif

	ld      rp1,rp1sv       ; restore application's rp1
	ret

;
; 24-2Fh: command codes
;       we use a series of 'djnz' tests (simpler & just as economical as
;       a jump table) to select the appropriate function routine
;
~cmd:   add     r11,#(30h-24h+1) ; readjust it upwards for djnz testing

;
; 24H: set a Forth "thread" breakpoint at the given address   (2 bytes)
;       this puts a the address of the "breakpoint" pseudo-word into
;       a Forth thread (a list of addresses of Forth words).  It is the
;       responsibility of the user to ensure that this is a valid thread
;       address, and to save the previous value.
;
~settb: djnz    r11,~setcb
	ld      r11,#^HB(tbreak)  ; store a pointer
	ldc     @rr14,r11
	ld      r11,#^LB(tbreak)  ; to the 'tbreak' Forth word
	ldc     1(rr14),r11
	jr      ~exit

;
; 25H: set a Forth "code field" breakpoint at the given address  (3 bytes)
;       this changes the code field associated with a given Forth word to
;       point to a machine-language breakpoint routine.  It is the
;       responsibility of the user to ensure that this is a valid code
;       field address, and to save the previous value.
;       >>>For the Super8 DTC Forth, this is the same as function 26H<<<
;
~setcb: djnz    r11,~setmb
	inc     r11             ; fall thru next djnz test

;
; 26H: set a machine language breakpoint at the given address  (3 bytes)
;       this puts a machine-language CALL at the given address.  It is
;       the responsibility of the user to ensure that this is a valid
;       instruction address, and to save the previous value.
;
~setmb: djnz    r11,~getlo
	ld      r11,#0f6h         ; build a 'call' instruction
	ldc     @rr14,r11
	ld      r11,#^HB(mbreak)  ; to the 'mbreak' entry point
	ldc     1(rr14),r11
	ld      r11,#^LB(mbreak)
	ldc     2(rr14),r11
	jr      ~exit

;
; 27H: copy low address byte to data register, and send to host
;       Note that this destroys the previous data register contents.
;       Use function 2Dh to save that value first, if needed.
;
~getlo: djnz    r11,~gethi
	ld      r12,r15         ; get low adrs byte in data reg
	jr      ~echo1          ; and go send it

;
; 28H: copy high address byte to data register, and send to host
;       Note that this destroys the previous data register contents.
;       Use function 2Dh to save that value first, if needed.
;
~gethi: djnz    r11,~echo
	ld      r12,r14         ; get high adrs byte in data reg
	jr      ~echo1          ; and go send it

;
; 29H: read back contents of data register
;
~echo:  djnz    r11,~fetch
~echo1: txon                    ; turn on transmitter (tx buf is empty)
	ld      r11,r12         ; transmit high nybble
	swap    r11
	and     r11,#0fh
	or      r11,#30h

	.if system<>3 ; not XP ----------------------------------------
	ld      uio,r11
~tx1:   tm      utc,#2          ; wait for tx buf to empty
	jr      z,~tx1

	ld      r11,r12         ; transmit low nybble
	and     r11,#0fh
	or      r11,#30h
	ld      uio,r11
~tx2:   tm      utc,#2          ; wait for tx buf to empty
	jr      z,~tx2

	.else ; XP ----------------------------------------------------
	ldc     sccadata,r11
	nop
	nop
	nop

~tx1:   ldc     r11,sccacmd
	tm      r11,#4          ; wait for tx buf to empty
	jr      z,~tx1

	ld      r11,r12         ; transmit low nybble
	and     r11,#0fh
	or      r11,#30h
	ldc     sccadata,r11
	nop
	nop

~tx2:   ld      r11,#1   ; wait until tx empty
	ldc     sccacmd,r11
	nop
	nop
	nop
	ldc     r11,sccacmd
	tm      r11,#1
	jr      z,~tx2
	.endif ; XP ---------------------------------------------------

	txoff                   ; turn off transmitter
	jr      ~exit

;
; 2AH: fetch byte from memory or register, and transmit
;
~fetch: djnz    r11,~store
	; the fetch operations can go one of five ways, depending on r13
	ld      r11,r13
~efetch: djnz   r11,~ifetch             ; 1: E memory
	 lde    r12,@rr14
	 jr     ~fdone
~ifetch: djnz   r11,~dfetch             ; 2: indirect reg
	 ld     r12,@r15
	 jr     ~fdone
~dfetch: djnz   r11,~f1                 ; 3: direct reg, b0
	 jr     ~f0
~f1:     djnz   r11,~cfetch             ; 4: direct reg, b1
	 sb1
~f0:     push   r14                     ;   save 'mar'
	 push   r15
	 ld     r14,r15                 ;   put reg adrs in 'mar'
	 ld     rp1,mar+1               ;   hi 5 bits of reg adrs -> rp1
	 and    mar,#7                  ;   lo 3 bits of reg adrs -> mar
	 mult   mar,#index              ;   calculate offset to read routine
	 add    mar+1,#^LB(dirrd)       ;   and add base address
	 adc    mar,#^HB(dirrd)
	 call   @mar                    ;   go read the register into mdr
	 srp1   #regs                   ;   restore talker's rp1
	 pop    r15                     ;   restore 'mar'
	 pop    r14
	 sb0
	 jr     ~fdone
~cfetch: ldc    r12,@rr14               ; 0: C memory
~fdone: incw    rr14
	; now output two hex digits from the data register just fetched
	jr      ~echo1

;
; 2BH: store byte in memory or register, per bank select
;
~store: djnz    r11,~setlo
	; the store operations can go one of five ways, depending on r13
	ld      r11,r13
~estore: djnz   r11,~istore             ; 1: E memory
	 lde    @rr14,r12
	 jr     ~sdone
~istore: djnz   r11,~dstore             ; 2: indirect reg
	 ld     @r15,r12
	 jr     ~sdone
~dstore: djnz   r11,~s1                 ; 3: direct reg, b0
	 jr     ~s0
~s1:     djnz   r11,~cstore             ; 4: direct reg, b1
	 sb1
~s0:     push   r14                     ;   save 'mar'
	 push   r15
	 ld     r14,r15                 ;   put reg adrs in 'mar'
	 ld     rp1,mar+1               ;   hi 5 bits of reg adrs -> rp1
	 and    mar,#7                  ;   lo 3 bits of reg adrs -> mar
	 mult   mar,#index              ;   calculate offset to read routine
	 add    mar+1,#^LB(dirwr)       ;   and add base address
	 adc    mar,#^HB(dirwr)
	 call   @mar                    ;   go write the register from mdr
	 srp1   #regs                   ;   restore talker's rp1
	 pop    r15                     ;   restore 'mar'
	 pop    r14
	 sb0
	 jr     ~sdone
~cstore: ldc    @rr14,r12               ; 0: C memory
~sdone: incw    rr14
	jr      ~nofunc

;
; 2CH: set low address byte
;
~setlo: djnz    r11,~sethi
	ld      r15,r12
	jr      ~nofunc

;
; 2DH: set high address byte
;
~sethi: djnz    r11,~setext
	ld      r14,r12
	jr      ~nofunc

;
; 2EH: set extended address byte (memory page)
;  Note: out of range values will default to C memory, later
;
~setext: djnz   r11,~go
	ld      r13,r12
~nofunc: jp     ~exit

;
; 2FH: go to given address (resume execution at given address)
;
~go:    djnz    r11,~nofunc
	incw    sph             ; drop return adrs in 'talker'
	incw    sph
	ld      flags,r12       ; restore saved flags (if any)
	ld      rp1,rp1sv       ; restore saved rp1
	.if system=3 ; XP
	fast
	.endif
	ei
	jp      @mar            ; go to address in monitor adrs reg (rr14)

;========================================================================
;
;       DIRECT REGISTER ACCESS ROUTINES
;
;       These routines are called to perform direct reads and writes
;       to working registers r8-r15, under program control.  These
;       are used to perform selectable, direct register access to
;       registers C0-FF by changing the register pointer RP1.
;
;       This is necessary to allow the host to access the Super8's
;       system and control registers.  The host supplies the address,
;       but indirect register reference cannot access these registers.
;       Instead, the register address is used to set RP1 and to select
;       one-of-eight direct access routines.
;
;========================================================================
index   .equ    3       ; number of bytes occupied by each routine,
			; used to calculate address from register #.

dirrd:  ld      mdr,r8  ; read register '0'
	ret
	ld      mdr,r9  ; read register '1'
	ret
	ld      mdr,r10 ; read register '2'
	ret
	ld      mdr,r11 ; read register '3'
	ret
	ld      mdr,r12 ; read register '4'
	ret
	ld      mdr,r13 ; read register '5'
	ret
	ld      mdr,r14 ; read register '6'
	ret
	ld      mdr,r15 ; read register '7'
	ret

dirwr:  ld      r8,mdr  ; write register '0'
	ret
	ld      r9,mdr  ; write register '1'
	ret
	ld      r10,mdr ; write register '2'
	ret
	ld      r11,mdr ; write register '3'
	ret
	ld      r12,mdr ; write register '4'
	ret
	ld      r13,mdr ; write register '5'
	ret
	ld      r14,mdr ; write register '6'
	ret
	ld      r15,mdr ; write register '7'
	ret

	.if system=3 && standalone ; XP
;========================================================================
;
;       Z8530 SCC INITIALIZATION - TABLES
;
;       This table is the "standard" initialization for the
;	Zilog Serial Communications Controller.  It follows
;	the recommended initialization sequence set forth in
;	the Zilog Z8530 SCC Technical Manual, Chapter 8.
;
;       This table sets up channel A for 9600 baud,
;	asynchronous operation with 8 bits, 1 stop, no parity.
;
;       This table disables all interrupts in the SCC.
;       (Interrupts will be enabled elsewhere, when used.)
;
;       Useful baud rate constants (x16 mode):
;          9600 baud = 6    1200 baud = 63
;
;========================================================================

sccaparms: ; modes and constants for PORT A
	.byte   0               ; just in case...reset reg pointer to 0
	.byte	9,11000000b	; hardware reset, irpts off
	.byte   4,01000100b     ; 16x clock, async 1 stop, no parity
	.byte   1,0             ; no dma, all irpts disabled
	.byte	2,0		; irpt vector (read under pgm control)
	.byte	3,11000000b	; rx 8 bits, rx disabled
	.byte	5,01100000b	; tx 8 bits, tx disabled, RTSA high
	.byte	9,00000001b	; status low, irpts off, vector incl status
	.byte	10,0		; nrz encoding
	.byte	11,01010000b	; no xtal, rx & tx clock from BRG, TRxC in
	.byte	12,6		; BRG lo byte - for 9600 baud in 16x mode,
	.byte	13,0		;		from 2.5 MHz BRG clock (RTxC)
	.byte	14,00000100b	; DTR pgm'd, BRG from RTxC, BRG disabled
	; enables
	.byte	14,00000101b	;   as above, plus BRG enabled
	.byte	3,11000001b	;   as above, plus rx enabled
	.byte	5,01101000b	;   as above, plus tx enabled
	; interrupts
	.byte   15,00000000b    ; no ext/sts irpts
	.byte	10h		; reset ext/sts irpts
	.byte	10h		; reset ext/sts irpts again
	.byte   1,00000000b     ; no dma, all irpts disabled

numaparms .equ $-sccaparms

;
;	SCCFILL - output a block of data to the SCC
;
;	Expects: rr8 = address of SCC port
;		 rr10 = address of data
;		 r13 = number of bytes
;	Returns: none
;	Uses:	 r8-r13
;
sccfill: ldci   r12,@rr10
	ldc     @rr8,r12        ; "LDC" on Echelon!
	djnz	r13,sccfill
	ret

	.endif ; XP
