;========================================================================
;
;       ZILOG Z8 "MICRO-TALKER" MONITOR PROGRAM
;       (c) 1990 T-Recursive Technology
;       placed into the public domain for free and unrestricted use
;
;       A minimal monitor program for the Zilog Z8.
;       Used in conjunction with a "smart" host program to examine &
;       modify code memory, external memory, and registers, and
;       to set and execute breakpoints in machine language and Forth.
;
;       The Talker program uses only 6 registers (0Ah to 0Fh here),
;       no RAM, and about 300 bytes of PROM.  No interrupts are used.
;       The Talker may be operated half-duplex over a bidirectional
;       serial line.
;
;       The program accepts characters from the Z8 UART.
;       Characters 30h to 3Fh are treated as hex digits and are
;       shifted into a one-byte data register.
;       Characters 20h to 2Fh are command codes:
;           20-23 reserved for future use (ignored)
;         group 2: breakpoint / debug support
;           24 = set a Forth "thread" breakpoint
;           25 = set a Forth "code field" breakpoint
;           26 = set a machine language breakpoint
;           27 = fetch lo byte of address (adrs lo -> data & output)
;           28 = fetch hi byte of address (adrs hi -> data & output)
;           29 = read back data register  (data -> output)
;         group 1: minimal talker
;           2A = fetch byte & incr addr   (mem or reg -> data & output)
;           2B = store byte & incr addr   (data -> mem or reg)
;           2C = set lo byte of address   (data -> adrs lo)
;           2D = set hi byte of address   (data -> adrs hi)
;           2E = set memory "page"        (data -> page)
;           2F = "go"                     (jump to current adrs)
;
;       The memory "page" is interpreted as follows for store and fetch:
;           0 = C (code) memory
;           1 = E (external) memory
;           2 = registers 00-FF
;
;       Internal register usage:
;           rr14 (0E,0F) = address
;           r13  (0D)    = memory "page"
;           r12  (0C)    = data byte
;           r11  (0B)    = working
;           r10  (0A)    = saved rp1
;
;       Revision history
;       v 1.0  23 Aug 89    original Super8 program; functions 29h - 2Fh;
;               access to memory, registers, indirect registers;
;               'talker' and 'talk' entry points.
;
;       v 1.1  25 Feb 90    support for breakpoints and Forth words;
;               standalone initialization.
;
;       v 1.2  7 May 90     function codes reassigned; added address
;               readback; improved breakpoint support; fixed problem
;               with direct register access to C8-CF and to RP1...now
;               correctly uses application program's registers.
;
;       v 1.2CP  7 Jun 90   modified for Teatronics Echelon Channel Proc,
;               as an include file.  Uses half-duplex RS-485.  Sets P37
;               low to turn LED 'on'.  Pulses watchdog on TXD.
;               Fixed bug where tx turned off during last tx character.
;               Changed to disable irpts on 'mbreak' entry, enable on 'go'.
;
;               26 Jun 90   altered to assemble standalone; set IMR to 00
;               so that irpt enable on 'go' doesn't lock up system.
;
;       Z8 v1.0 2 Dec 90  modified for Zilog Z8
;
;========================================================================
;
;       Monitor Configuration and Assembly-Time Options
;
;       STANDALONE - set to '1' if this file is to be assembled as a
;           standalone program, to be put into PROM.  Set to '0' if
;           this file is to be 'included' in another source file.
;
standalone .equ 1

;
;       Macros for half-duplex communication on a bidirectional link,
;       e.g., a bidirectional RS-485 serial line.  Define these macros
;       to control the transceiver connected to the Z8's serial port.
;       If full-duplex is to be used (separate transmit and receive data
;       lines), define these as "null" macros.
;
;       TXON - turns serial port transmitter on, and receiver off.
;       TXOFF - turns serial port transmitter off, and receiver on.
;       KICK - kick the watchdog timer (v. 1.2CP)
;
txon    .macro
	.endm

txoff   .macro
	.endm

kick    .macro
	.endm

;
;       Register bank used by monitor: the monitor requires 6 bytes
;       of Z8 registers.  These must be the last 6 bytes of a
;       16-byte register bank, since they will be mapped (via RP) onto
;       working registers R10-R15.  These will be accessed directly
;       as working registers AND as general purpose registers.
;       Set 'regs' to an 16-byte boundary between 00 and 70, inclusive.
;
regs:   .equ    00h    ; monitor will use regs+0b to regs+0f (0B to 0F)

rp1sv:  .equ    regs+8+2  ; save area for applic's rp
wkg:    .equ    regs+8+3  ; working "scratch" register    aka r11
mdr:    .equ    regs+8+4  ; memory data register          aka r12
mbr:    .equ    regs+8+5  ; memory bank register          aka r13
mar:    .equ    regs+8+6  ; memory address register pair  aka rr14

;========================================================================
;
;       STANDALONE INITIALIZATION
;
;       For use when the monitor is used as a standalone program
;       in a Z8 development board.  In this case, the monitor
;       is located in low PROM, to be started on reset.  The Z8
;       registers are "minimally" initialized, to allow full-duplex
;       serial communication at 4800/9600 baud, 8 bits, no parity.
;       The interrupts are vectored to a supplementary jump table
;       in code RAM.
;
;       Expects:  reset state for all Super8 mode & control registers
;       Returns:
;       Uses:
;
;========================================================================
	.if standalone
;
;       interrupt vectors
;
vecs    .equ    0e000h  ; base address of the interrupt jump table

	.org    0       ; the Z8 interrupt vector table
	.word   vecs    ; irq0  p32
	.word   vecs+3  ; irq1  p33
	.word   vecs+6  ; irq2  p31, Tin
	.word   vecs+9  ; irq3  p30, serial in
	.word   vecs+12 ; irq4  T0, serial out
	.word   vecs+15 ; irq5  T1
;
;       z8 initialization...immediately follows vectors
;
	clr     p0              ; output 0 hi adrs bits, just in case
	ld      p01m,#10110010b ; p0=a8-a15, p1=ad0-ad7, ext'l stack, slow mem

	clr     p3              ; output 0 for p34 (DM\)
	ld      p3m,#01000001b  ; p34=out(0) p33=in; p35=out p32=in
				; p36=Tout p31=Tin; p37=Sout p30=Sin
				; p2 push-pull; parity off
	ld      p2m,#0ffh       ; p2 all input

	ld      ipr,#00010001b  ; arbitrary irpt priority
	clr     imr             ; all irpts disabled
	ei                      ; must 'ei' to enable IRQ register!
	di                      ; then we can 'di'
	or      irq,#10h        ; set 'tx ready' bit in IRQ for 1st byte

	ld      pre0,#(13*4)+1  ; prescale=13; modulo N timer
	ld      t0,#1           ; for 4800 baud @ 8 MHz clock
	ld      tmr,#00000011b  ; Tout pgmd; Tin clk; t1 off; load & go t0
	; pre1 and t1 not initialized at this time

	ld      spl,#0ffh       ; stack at top of RAM
	ld      sph,#0ffh
	srp     #10h            ; wkg regs are 10-1f
	; ei

	jr      talker

	.endif ; standalone

;========================================================================
;
;       TALKER MAIN ENTRY POINT
;       MACHINE LANGUAGE BREAKPOINT ENTRY
;
;       This is the main "talker" program.  It calls the basic
;       character processing routine "talk" repeatedly, until a monitor
;       command transfers control to an application program.
;
;       This is also the entry point for machine language breakpoints.
;       A breakpoint consists of a "CALL" to this address.  The call
;       causes the calling address to be pushed on the stack; this
;       routine pushes the flags as well, to allow a common routine to
;       service both CALLs and interrupts.
;
;       Should it be desirable to have an interrupt cause a breakpoint
;       -- e.g., a "break" pushbutton wired to an interrupt input --
;       the alternate "ibreak" entry point can be used.
;
;       The breakpoint routine copies the saved flag values into the
;       talker's data register, and the saved return address into the
;       talker's address register.  An immediate "go" function will
;       resume with these saved values (as well as the saved rp).
;
;       Entering the breakpoint routine causes the '*' character to
;       be sent to the host.  Entering the monitor causes 'M' to be sent.
;       Note also that the main entry point its own address onto
;       the stack; this is so that we can "go" to a routine which ends
;       in a RET.  This "normal termination" can be distinguished from
;       a breakpoint by the 'M' character.
;
;       Expects:
;       Returns:
;       Uses:   4 bytes of stack
;
;========================================================================
	.begin
~pushme: call   ~t0
talker: ld      wkg,#'M' ; Talker program entry point
	jr      ~pushme     ; some cleverness to push the address 'talker'

mbreak: push    flags   ; Machine language breakpoint entry
	di
ibreak:                 ; Interrupt breakpoint entry
	pop     mdr         ; get saved flags in mdr
	pop     mar         ; get saved adrs in mar (2 bytes)
	pop     mar+1
	ld      wkg,#'*'

	; All entry points eventually land here
~t0:    ld      rp1sv,rp   ; save application's rp value

~t1:    tm      irq,#10h ; transmitter ready for another character?
	jr      z,~t1
	and     irq,#0efh ; clear the irq bit
	push    wkg      ; save the 'M' or '*' character during 'txon'
	txon
	pop     sio      ; transmit 'M' or '*' to signal monitor/brkpt
~t2:    tm      irq,#10h ; wait 'til character finished
	jr      z,~t2
	txoff

~t3:    kick            ; at this point tx is off, so kick TXD  v. 1.2CP
	call    talk    ; the talker loop
	jr      ~t3
	.end

;========================================================================
;
;       FORTH LANGUAGE BREAKPOINT ENTRIES
;
;       These are the entry points for Forth language breakpoints.
;
;       The first is the "code field" breakpoint.  This is an address
;       which can be stored in a Forth word's code field, to cause a
;       break whenever that word is executed.  This kind of breakpoint
;       can be set in any Forth word.
;       >>> In the Z8 Direct-Threaded-Code implementation, the
;         parameter field of every Forth word begins with machine code.
;         So, an ordinary machine-code breakpoint can be set in the
;         first 3 bytes of a word.  (All words have at least 3 bytes.) <<<
;
;       The second kind is the "thread" breakpoint.  This is an address
;       which can be patched into a high-level "thread", to cause a
;       break when a certain point is reached in high-level code.  The
;       thread is a series of addresses of Forth words, executed by the
;       inner or "NEXT" interpreter.  So, we provide the address of a
;       dummy Forth word whose execution action is to invoke a machine
;       language breakpoint.  The Forth execution state can be deduced
;       from the registers.
;
;========================================================================
cbreak: .equ    mbreak  ; the code field breakpoint is simply
			; a machine language breakpoint set
			; in a Z8 DTC "code field"

; what follows is the parameter and code field of a "headerless"
; Forth word, to invoke the breakpoint routine.  (In DTC, this is
; simply a machine code fragment which does a breakpoint.)

tbreak: call    mbreak  ; the thread breakpoint is simply
			; a pointer to this code fragment

;========================================================================
;
;       TALK - SINGLE CHARACTER PROCESSING ROUTINE
;
;       This routine processes one character received from the host.
;       If no character is received, it returns immediately.
;       It may cause two characters to be transmitted to the host.
;
;       This routine is called repeatedly in a tight loop from the
;       'talker' program, if invoked standalone or by a breakpoint.
;       It may also be called repeatedly from within an application
;       program, as a polled "background" task, to perform monitor
;       functions simultaneously with the application.
;
;       Note - on entry, the stack contains the following:
;                       return adrs in 'talker', lo
;                SP-->  return adrs in 'talker', hi
;
;       Expects:
;       Returns: rp1 = 'regs'
;       Uses:    rp1, flags, regs BBh-BFh.
;
;========================================================================
	.begin
talk:                           ; check for received character
	tm      irq,#8
	jr      z,~exit

	srp     #regs           ; point r8-r15 to talker's regs

	and     irq,#0f7h       ; clear the irq bit
	ld      r11,sio         ; get character
	and     r11,#3fh        ; mask off all but low 6 bits
	sub     r11,#30h        ; if less than 30 -
	jr      ult,~cmd        ; - it's a command

;
; 30-3Fh: digit entry
;
~digit: swap    r12             ; 30-3f: hex digit, shift into lo nybble
	and     r12,#0f0h
	add     r12,r11         ;   note that r11 has been converted to 00-0f

~done:
~exit:  ret

;
; 24-2Fh: command codes
;       we use a series of 'djnz' tests (simpler & just as economical as
;       a jump table) to select the appropriate function routine
;
~cmd:   add     r11,#(30h-24h+1) ; readjust it upwards for djnz testing

;
; 24H: set a Forth "thread" breakpoint at the given address   (2 bytes)
;       this puts a the address of the "breakpoint" pseudo-word into
;       a Forth thread (a list of addresses of Forth words).  It is the
;       responsibility of the user to ensure that this is a valid thread
;       address, and to save the previous value.
;
~settb: djnz    r11,~setcb
	ld      r11,#^HB(tbreak)  ; store a pointer
	ldc     @rr14,r11
	ld      r11,#^LB(tbreak)  ; to the 'tbreak' Forth word
	incw    rr14
	ldc     @rr14,r11
	decw    rr14
	ret

;
; 25H: set a Forth "code field" breakpoint at the given address  (3 bytes)
;       this changes the code field associated with a given Forth word to
;       point to a machine-language breakpoint routine.  It is the
;       responsibility of the user to ensure that this is a valid code
;       field address, and to save the previous value.
;       >>>For the Super8 DTC Forth, this is the same as function 26H<<<
;
~setcb: djnz    r11,~setmb
	inc     r11             ; fall thru next djnz test

;
; 26H: set a machine language breakpoint at the given address  (3 bytes)
;       this puts a machine-language CALL at the given address.  It is
;       the responsibility of the user to ensure that this is a valid
;       instruction address, and to save the previous value.
;
~setmb: djnz    r11,~getlo
	ld      r11,#0d6h         ; build a 'call' instruction
	ldc     @rr14,r11
	ld      r11,#^HB(mbreak)  ; to the 'mbreak' entry point
	incw    rr14
	ldc     @rr14,r11
	ld      r11,#^LB(mbreak)
	incw    rr14
	ldc     @rr14,r11
	decw    rr14
	decw    rr14
	ret

;
; 27H: copy low address byte to data register, and send to host
;       Note that this destroys the previous data register contents.
;       Use function 2Dh to save that value first, if needed.
;
~getlo: djnz    r11,~gethi
	ld      r12,r15         ; get low adrs byte in data reg
	jr      ~echo1          ; and go send it

;
; 28H: copy high address byte to data register, and send to host
;       Note that this destroys the previous data register contents.
;       Use function 2Dh to save that value first, if needed.
;
~gethi: djnz    r11,~echo
	ld      r12,r14         ; get high adrs byte in data reg
	jr      ~echo1          ; and go send it

;
; 29H: read back contents of data register
;
~echo:  djnz    r11,~fetch
~echo1: txon                    ; turn on transmitter (tx buf is empty)
	ld      r11,r12         ; transmit high nybble
	swap    r11
	and     r11,#0fh
	or      r11,#30h
	and     irq,#0efh ; clear the irq bit
	ld      sio,r11
~tx1:   tm      irq,#10h        ; wait for tx buf to empty
	jr      z,~tx1

	ld      r11,r12         ; transmit low nybble
	and     r11,#0fh
	or      r11,#30h
	and     irq,#0efh ; clear the irq bit
	ld      sio,r11
~tx2:   tm      irq,#10h        ; wait for tx buf to empty
	jr      z,~tx2
	txoff                   ; turn off transmitter
	ret

;
; 2AH: fetch byte from memory or register, and transmit
;
~fetch: djnz    r11,~store
	; the fetch operations can go one of five ways, depending on r13
	ld      r11,r13
~efetch: djnz   r11,~ifetch             ; 1: E memory
	 lde    r12,@rr14
	 jr     ~fdone
~ifetch: djnz   r11,~cfetch             ; 2: indirect reg
	 ld     r12,@r15
	 jr     ~fdone
~cfetch: ldc    r12,@rr14               ; 0: C memory
~fdone: incw    rr14
	; now output two hex digits from the data register just fetched
	jr      ~echo1

;
; 2BH: store byte in memory or register, per bank select
;
~store: djnz    r11,~setlo
	; the store operations can go one of five ways, depending on r13
	ld      r11,r13
~estore: djnz   r11,~istore             ; 1: E memory
	 lde    @rr14,r12
	 jr     ~sdone
~istore: djnz   r11,~cstore             ; 2: indirect reg
	 ld     @r15,r12
	 jr     ~sdone
~cstore: ldc    @rr14,r12               ; 0: C memory
~sdone: incw    rr14
	ret

;
; 2CH: set low address byte
;
~setlo: djnz    r11,~sethi
	ld      r15,r12
	ret

;
; 2DH: set high address byte
;
~sethi: djnz    r11,~setext
	ld      r14,r12
	ret

;
; 2EH: set extended address byte (memory page)
;  Note: out of range values will default to C memory, later
;
~setext: djnz   r11,~go
	ld      r13,r12
~nofunc: ret

;
; 2FH: go to given address (resume execution at given address)
;
~go:    djnz    r11,~nofunc
	incw    sph             ; drop return adrs in 'talker'
	incw    sph
	ld      flags,r12       ; restore saved flags (if any)
	ld      rp,rp1sv       ; restore saved rp1
	ei
	jp      @mar            ; go to address in monitor adrs reg (rr14)
