\ FIGURE 2. THE HOST PROGRAM

\ SUPER8 "MICRO-TALKER" host program for IBM PC
\   (c) 1989,1990 T-Recursive Technology
\   placed into the public domain for free and unrestricted use
\
\ This is the host program for the minimal serial monitor
\ program, with customization for the Zilog Super8.
\
\ vers 1.0  original program under real-Forth for the IBM PC
\
\ vers 1.1  26 Jun 90                  bjr
\           modified talker function codes; added breakpoint
\           functions; various minor improvements
\
\ vers 1.2  2 Jul 90                   bjr
\           converted from screen file to text file; modified
\           for L.O.V.E.-83Forth; added multiple breakpoints;
\           various minor improvements
\
\ vers 1.3  10 Oct 90                  bjr
\           converted for 6-port SIO card using 2681, (2) Z8530
\
\ vers 1.4  17 Dec 90                  bjr
\           modified for MPE PowerForth; added support for
\           COM1 thru COM4
\
\ vers 1.4TCJ  26 May 91               bjr
\           removed support for 6-port SIO card and COM2:-COM4:;
\           removed Super8-specific words, for TCJ article
\
VOCABULARY TALKER  ONLY FORTH ALSO TALKER DEFINITIONS

\ ******************** BASIC SERIAL I/O ************************
HEX

\ port select parameters
VARIABLE 'STSREG       \ status register, for multiport words
VARIABLE 'DTAREG       \ data register, for multiport words

3F8 CONSTANT COM1PORT  \ COM1: base address (data reg, sts is at 5 + )
2F8 CONSTANT COM2PORT  \ COM2:
3E8 CONSTANT COM3PORT  \ COM3:
2E8 CONSTANT COM4PORT  \ COM4:

: COM1:   COM1PORT 5 + 'STSREG !  COM1PORT 'DTAREG !  ;

\ ?TX  returns true if transmitter is ready for a character
: ?TX  'STSREG @ PC@ 20 AND ;      \ COM ports

\ ?RX  returns true if receiver has a character
: ?RX  'STSREG @ PC@ 1 AND ;       \ DUART, SCC, and COM ports

\ TXON  uses the DTR line to switch a remote 75176 RS-485
\  transceiver to the "transmit" mode.
: TXON   0 'DTAREG @ 4 + PC! ;                 \ COM ports

\ TXOFF  switches the remote transceiver to the "receive" mode.
\  It waits for the last character to clear the tx.
: TXOFF   BEGIN  'STSREG @ PC@  40 AND UNTIL  3 'DTAREG @ 4 + PC! ;

\ (TX)  transmits a character thru the SIO port.  Note that it
\  DOES NOT wait for the UART to be ready.
: (TX)  'DTAREG @ PC! ;    \ all devices

\ (RX)  gets a character from the SIO port.  Note that it
\  DOES NOT wait for the UART to be ready.
: (RX)  'DTAREG @ PC@ ;    \ all devices

\ PACE is used to slow the PC down for slow target CPUs
VARIABLE PACE  DECIMAL  \ approx 1.5 usec per count w/ 9.54 MHz V30
: FAST        1 PACE ! ;
: MEDIUM    500 PACE ! ;
: SLOW    10000 PACE ! ;
: PACED   PACE @ 0 DO LOOP ;
HEX

\ ****************  UART INITIALIZATION  ********************

CREATE BAUDTABLE  ( COMn:, 4800 baud, N-8-1 )
   6 C,  \ 6 pairs follow
   3F8 1 + ,  0 C,   \ disable uart irpts
   3F8 3 + ,  80 C,  \ enable divisor latch
   3F8 ,      18 C,  \ divisor low
   3F8 1 + ,  0 C,   \ divisor high
   3F8 3 + ,  3 C,   \ 8 bits, no parity, 1 stop
   3F8 4 + ,  3 C,   \ modem ctl: RTS, DTR

\ General initialization program...expects a count, followed
\ by a list of  port # (low byte of port adrs), data  pairs
: S-INIT ( a )
   BAUDTABLE   DUP C@ 0 ?DO
      1+ DUP 2+ DUP  C@ ( data ) ROT @ ( port ) PC!
   LOOP DROP ;

\ ************** QUICK & DIRTY TERMINAL PROGRAM ****************
HEX
: COLOR   CURR-ATTRIBS ! ;     \ change screen color in PForth
07 constant white
70 constant yellow   \ actually this is black on white

\ We redefine TX to use the half-duplex serial link.  It turns
\ the transmitter on before sending, sends, then waits for the
\ character to finish and turns the transmitter off.
\ If using a full duplex link, this definition can be omitted.
: TRAP      KEY? IF ." *escape*" KEY DROP QUIT THEN ;
: TX ( c )   PACED  BEGIN TRAP ?TX UNTIL  TXON (TX) TXOFF ;
: RX ( - c )  BEGIN TRAP ?RX UNTIL  (RX) ;

\ TERM  is a simple terminal program.  ESC exits back to Forth.
: (TERM)   BEGIN   ?RX IF  (RX) EMIT  THEN
      ?TX IF  KEY? IF  KEY DUP
	 1B = IF  DROP EXIT  ELSE  TXON (TX) TXOFF  THEN
      THEN THEN
   AGAIN ;

: TERM   YELLOW COLOR  (TERM)  WHITE COLOR ;

\ **************** SUPPORT FUNCTIONS FOR TALKER ****************
HEX
\ SPILL  will empty the UART receiver (read and discard chars
\  until the UART is empty).  This is frequently necessary,
\  e.g., when external serial switches are used, or if the
\  target malfunctions.  ANY garbage characters in the rx data
\  stream will "unsynchronize" the talker and cause permanent
\  confusion, so it's good to SPILL at frequent intervals.
: SPILL   BEGIN ?RX WHILE RX DROP REPEAT ;

\ NYBLIZE  takes an 8-bit value, and converts it to two
\  pseudo-hex characters (ASCII characters from 30 to 3F hex).
: NYBLIZE ( c - lo hi )  0FF AND 10 /MOD  SWAP 30 + SWAP 30 + ;

\ DENYBL  takes two pseudo-hex characters (30 to 3F hex), and
\  converts them to a byte value (00 to FF).
: DENYBL  ( hi lo - c )   0F AND SWAP 0F AND 10 * OR ;

\ ><  swaps the hi and lo bytes of the top stack item.
CREATE SWAPPER  3 ALLOT
: >< ( n - n )   SWAPPER !  SWAPPER C@ SWAPPER 2+ C!  SWAPPER 1+ @ ;

\ >BYTES  splits the top stack item into its high and low bytes
\ >WORD  takes high and low byte values and merges them into
\  a 16-bit word value.
: >BYTES ( n - lo hi )   DUP >< ;   ( note hi 8 bits unknown )
: >WORD ( hi lo - n )   SWAP >< OR ;   ( hi 8 bits must be 0 )

\ TXH  transmits a byte as two pseudo-hex characters.
\ RXH  receives a byte as two pseudo-hex characters.
: TXH ( n )   NYBLIZE TX TX ;    ( transmit a hex value )
: RXH ( - n )   RX RX DENYBL ;   ( receive a hex value )

\ **************** PRIMITIVE TALKER OPERATIONS *****************
HEX
\ XADR  sends the "current address" to the target.
\  This is done as: send hi byte, send "hi adrs" command,
\  send lo byte, send "lo adrs" command.
\  We do a SPILL as part of XADR because XADR is used by
\  everything, and when it is used we aren't expecting data.
: XADR ( a )   SPILL >BYTES  TXH 2D TX ( hi )  TXH 2C TX ( lo ) ;

\ MPGE  builds words which send a page-select command to the
\  target.  Pages are selected by an 8-bit "page address."
\ CMEM, EMEM, and REGS are the memory pages defined
\  for the Z8 talker program.
: MPGE   CREATE  C,  DOES> C@ TXH  2E TX ;
0 MPGE CMEM    \ "C" code memory
1 MPGE EMEM    \ "E" external memory
2 MPGE REGS    \ registers

\ X@+  fetches a byte from the target at the current address.
\ X!+  stores a byte in the target at the current address.
\  Both of these functions use the currently selected memory
\  page, and increment the current address when done.
: X@+ ( - n )   2A TX  RXH ;
: X!+ ( n )   TXH  2B TX ;

\ JUMP  starts target execution at the current address.
\  The Z8 talker expects the flag byte to be in the talker
\  data register when a "go" is issued.  This words sends a zero
\  flag byte, which will force bank 0 (also no fast irpt).
: JUMP ( a )   XADR  0 TXH 2F TX ;

\ *************** MEMORY/REGISTER DUMP AND EDIT ****************
HEX
\ ?EMIT  filters nonprintable characters.
: ?EMIT ( c )  7F AND DUP 20 < IF DROP ASCII . THEN EMIT ;

\ DUMP  dumps a range of memory or registers.
: DUMP ( a n )   OVER + SWAP DO   CR I 5 U.R  2 SPACES
      I XADR  I 10 OVER + SWAP DO  X@+ 3 .R  LOOP  2 SPACES
      I XADR  I 10 OVER + SWAP DO  X@+ ?EMIT  LOOP
   10 +LOOP ;

\ FILL  fills a range of memory with a byte value c.
: FILL ( a n c )   ROT XADR  SWAP 0 DO  DUP X!+  LOOP  DROP ;

\ SET  is an interactive examine/alter of memory or registers,
\  modelled after the Zilog Super8 monitor "set" command.
\  Given an address, it displays its contents.  The operator
\  may then type <CR> to advance to the next address, or a
\  Forth expression followed by <CR> to store a new value and
\  advance.  The loop is exited with Q <CR> .
: SET ( a )   BEGIN  CR DUP 5 U.R   DUP XADR X@+ 3 .R  SPACE
   DEPTH >R QUERY INTERPRET   ( parse a line of Forth )
   DEPTH R> - 1 - 0= IF  ( if depth is +1, ... adr n on stack )
      OVER XADR  X!+
   THEN 1+ AGAIN ;

: Q ( a )  DROP QUIT ;  ( drop the adrs & return to Forth )

\ ******************** BREAKPOINT FUNCTIONS ********************

\ BPARRAY  is used to "remember" up to 10 breakpoints.  For each
\  we must remember the 2-byte address of the breakpoint, and
\  the 3 bytes at that address which were replaced by the
\  breakpoint CALL instruction.  We allot 3 cells (6 bytes).
\ BPADR  returns the "address storage" location for breakpt n.
\  A breakpoint is flagged 'inactive' by storing an address of
\  zero.  (So, you can't set breakpoints at address zero.)
\ BPMEM  returns the "target memory storage" location for b.p. n
CREATE BPARRAY   DECIMAL 10 6 * ALLOT
: BPADR ( n - adr )   6 * BPARRAY + ;
: BPMEM ( n - adr )   6 * BPARRAY + 2 + ;

\ INITBP  initializes all breakpoints (clears the breakpoints
\  without attempting to restore memory contents).
: INITBP   10 0 DO  0 I BPADR !  LOOP ;
INITBP

\ .BP  prints the addresses of all the currently active
\  (non-zero) breakpoints.
: .BP   CR ." Breakpoints are set at:"  10 0 DO
      I BPADR @  ?DUP IF  CR I 3 .R  7 U.R  THEN
   LOOP ;

\ FINDBP  finds the breakpoint with the given address.
\  If none found, -1 is returned.
: FINDBP ( adr - n )   -1  10 0 DO
      OVER I BPADR @ =  IF DROP I LEAVE THEN
   LOOP SWAP DROP ;

HEX
\ !BREAK  sets breakpoint n at the given address.
\  Note that this selects CMEM.
: !BREAK ( adr n )   2DUP BPADR !  ( save the address )
   CMEM OVER XADR  X@+ X@+ X@+   ( get old contents )
   >WORD  ROT BPMEM 2!           ( save old contents )
   XADR 26 TX ;         ( send adrs, request breakpoint )

\ -BREAK  restores the previous contents of breakpoint n,
\  and marks breakpoint n as inactive.
\  If breakpoint n was not set, no action is taken.
\  Note that this may select CMEM.
: -BREAK ( n )   DUP BPADR @ ?DUP IF   ( only if active! )
      CMEM XADR            ( set the address in the target )
      DUP BPMEM 2@ >BYTES ROT  ( get old contents, reversed )
      X!+ X!+ X!+          ( restore old contents )
      0 OVER BPADR !       ( mark this b.p. cleared )
   THEN DROP ;

\ BREAK sets breakpoint 'n' at the given address.
\  The previous breakpoint 'n' (0-9) is cleared.
\  If the new breakpoint address is zero, this
\  simply clears breakpoint 'n'.
: BREAK ( adr n )  DUP -BREAK  OVER IF !BREAK ELSE 2DROP THEN ;

\ BREAKPOINT  is used to build BP0 thru BP9.
\ BP0 thru BP9  are simply shorthand versions of
\  "0 BREAK" thru "9 BREAK", respectively.
: BREAKPOINT ( n )   CREATE C, DOES> C@ BREAK ;
0 BREAKPOINT BP0  1 BREAKPOINT BP1  2 BREAKPOINT BP2
3 BREAKPOINT BP3  4 BREAKPOINT BP4  5 BREAKPOINT BP5
6 BREAKPOINT BP6  7 BREAKPOINT BP7  8 BREAKPOINT BP8
9 BREAKPOINT BP9

\ ?BREAK  gets the target program status upon occurence of a
\  breakpoint.  The Z8, on a break, leaves the return
\  address in the talker address register, and the CPU flags
\  in the talker data register.  Note that the flags must be
\  read first, since all other operations destroy the data reg.
\  Note also that the breakpoint address is 3 less than the
\  return address (a Z8 CALL is 3 bytes).
\  The address and flags are saved in variables for RESUME.
VARIABLE RESUMEADR   VARIABLE RESUMEFLAG
: ?BREAK   29 TX RXH ( flags )  RESUMEFLAG !
   28 TX RXH ( ahi ) 27 TX RXH ( alo ) >WORD  3 - RESUMEADR !
   CR ." * BREAKPOINT " RESUMEADR @ FINDBP
   DUP 0< IF  ." unknown" DROP  ELSE  1 .R  THEN
   ."  at " RESUMEADR @ U.
   ."  flags=" RESUMEFLAG @ 2 .R ;

\ AWAIT  waits for a character to be received from the target.
\  If it is an '*', get and print the breakpoint info.
\  The talker sends an 'M' when the monitor is initialized,
\  an '*' when breakpoint encountered.
: AWAIT   RX ASCII * = IF ?BREAK THEN ;

\ GO  starts execution at a given address, and AWAITs.
: GO ( adr )   JUMP AWAIT ;

\ RESUME  continues execution after a breakpoint, and AWAITs.
\  Execution continues at the address of the breakpoint call.
\  NOTE! if you don't reset the breakpoint, you simply
\  trip it again instantly!
: RESUME   RESUMEADR @ XADR  RESUMEFLAG @ TXH  2F TX  AWAIT ;

\ ******************* HEX FILE LOAD AND SAVE *******************
HEX
PCB HEXFILE
VARIABLE CSUM        \ used for checksum calculation
VARIABLE HEXCHAR     \ can only READ and WRITE from memory!

\ hex-to-ASCII conversion table
CREATE HEXASC  30 C, 31 C, 32 C, 33 C, 34 C, 35 C, 36 C, 37 C,
     38 C, 39 C, 41 C, 42 C, 43 C, 44 C, 45 C, 46 C,

\ PUT  writes a character to the currently open file
\ PUTH  puts a single hex digit
\ PUTHH  puts a byte value as 2 hex digits, hi & lo
: PUT ( c )   HEXCHAR C!  HEXCHAR 1 HEXFILE HANDLE WRITE-PATH
   ABORT" Error writing file." DROP ;
: PUTH ( n )   HEXASC + C@ PUT ;
: PUTHH ( n )   DUP CSUM +!  DUP 0F0 AND 10 / PUTH  0F AND PUTH ;

\ WREC  writes a single record (line) of an Intel hex file,
\  starting at memory address a, for n bytes.  The record type
\  is typ.  If 0 bytes are requested, only the header info is
\  written.
: WREC ( typ a n )   0 CSUM !  ASCII : PUT ( rec start )
   DUP PUTHH ( length )  OVER >< PUTHH  OVER PUTHH ( adrs hi,lo )
   ROT PUTHH ( type )   DUP IF ( length>0 )
      OVER 0D EMIT 4 U.R
      SWAP XADR  0 DO  X@+ PUTHH  LOOP    ( data bytes )
   ELSE 2DROP THEN
   CSUM @ NEGATE PUTHH ( checksum )   0D PUT 0A PUT ( cr,lf ) ;

\ SAVE  given an address and length, writes it as a hex file.
\  The hex records will be no more than 16 data bytes long.
\  Note: maximum SAVE length is 32K.
: SAVE ( a n )   CMEM  HEXFILE PATHNAME "" .HEX" HEXFILE SET-EXT
   HEXFILE CREATE-PATH-PCB
   ABORT" Can't create file."  CR
   BEGIN  DUP 0>  WHILE
      2DUP 10 MIN  0 ROT ROT  WREC  ( write <= 16 bytes )
      SWAP 10 + SWAP 10 - REPEAT    ( +address, -length )
   2DROP  1 0 0 WREC
   HEXFILE HANDLE CLOSE-PATH ABORT" Can't close file." ;

\ GET  reads a character from the currently open file
\ GET:  scans the file until a : is encountered
\ GETH  gets a hex digit from the file
\ GETHH  gets a byte value from the file as two hex digits
: GET ( - c )   HEXCHAR 1 HEXFILE HANDLE READ-PATH
   ABORT" Error reading file."
   0= ABORT" End of file encountered."  HEXCHAR C@ ;
: GET:   BEGIN GET ASCII : = UNTIL ;
: GETH ( - n )   GET DUP 3F > IF 9 + THEN 0F AND ;
: GETHH ( - n )   GETH GETH DENYBL  DUP CSUM +! ;

\ GETREC  reads an Intel hex record from the file.  The
\  record type is returned on the stack.
: GETREC ( - typ )   0 CSUM !  GET:  GETHH ( length )
   GETHH >< GETHH + ( adrs )  DUP XADR  0D EMIT 4 U.R
   GETHH ( type )  SWAP
   ?DUP IF 0 DO  GETHH X!+  LOOP THEN   ( data bytes )
   CSUM @ GETHH + 0FF AND ABORT" Checksum error in hex file." ;

\ LOAD  loads a hex file into the target's memory.
: LOAD   CMEM  HEXFILE PATHNAME "" .HEX" HEXFILE SET-EXT
   HEXFILE OPEN-PATH-PCB
   ABORT" Can't open file."   CR
   BEGIN GETREC UNTIL    ( continue until record type <> 0 )
   HEXFILE HANDLE CLOSE-PATH ABORT" Can't close file." ;

ONLY FORTH ALSO TALKER
COM1:  S-INIT  FAST
.( Talker is ready.) CR
